"""Compute the gauge potential A_μ from flip counts and kernels.

This module defines two helper functions for computing the fractal
dimension ``D`` and the pivot weight ``g(D)``.  These are used by
``discrete_gauge.run_discrete_gauge`` to build the gauge potential A.

The actual computation of A (multiplying ``g(D)`` by kernels) is performed
in the discrete-gauge driver rather than here.  The functions in this
module are intentionally small and dependency‑free.
"""

import numpy as np


def logistic_D(n: np.ndarray, k: float, n0: float) -> np.ndarray:
    """Compute the fractal dimension D(n) using a logistic function.

    Parameters
    ----------
    n : np.ndarray
        Array of flip counts per link.
    k : float
        Logistic slope.  Larger values make the transition sharper.
    n0 : float
        Logistic center.  Values of n around n0 correspond to the midpoint of the curve.

    Returns
    -------
    np.ndarray
        Array of the same shape as ``n`` giving the dimension D for each link.
    """
    # To avoid overflow in the exponential for large k*(n-n0), clip the argument.
    x = np.clip(k * (n - n0), -50.0, 50.0)
    return 1 + 2 / (1 + np.exp(x))


def linear_gD(D: np.ndarray, a: float, b: float) -> np.ndarray:
    """Compute the pivot weight g(D) as a linear function of D.

    Parameters
    ----------
    D : np.ndarray
        Fractal dimension array.
    a : float
        Slope of the linear mapping.
    b : float
        Offset of the linear mapping.

    Returns
    -------
    np.ndarray
        The pivot weight g(D) for each link.
    """
    return a * D + b